-- World of Warcraft Addon
-- TargetNotes
-- author:  Tokipin@Mannoroth

--[[ ## slash commands ## ]]--

-- notetaking
local command1 = "/tn";
local command2 = "/df";

-- note searching
local search_command1 = "/ts";

_G["BINDING_HEADER_Toki's TargetNotes"] = "Toki's TargetNotes";

-- dummy settings table, to be properly used if an options UI is made
TARGETNOTES_SETTINGS = TARGETNOTES_SETTINGS or {};
TARGETNOTES_SETTINGS["max notes to display"] = 3;
TARGETNOTES_SETTINGS["highlight notes?"] = true;

--[[ ## internalize globals ## ]]--

CreateFrame( "frame", "TARGETNOTES_FRAME", UIParent, "frame:window-template" );

local tabulate_data = TARGETNOTES_FUNCTIONS["tabulate_data"];
local SearchWindow = TARGETNOTES_SEARCH;
local NoteDisplay = _G["TARGETNOTES_FRAME"];
local Highlight = _G["TARGETNOTES_FRAME:highlight"];
local Ping = _G["TARGETNOTES_FRAME:ping"]; -- the little red thingie
local DB;

local highlight_notes_q = function()
    if( TARGETNOTES_SETTINGS["highlight notes?"] ) then
        Highlight:Show();
    else
        Highlight:Hide();
    end
end

-- timer for flashing little red thing on n off
local timer = CreateFrame( "frame", nil, UIParent, "frame:timer" );

local active_server;

local unit_properties = {
    ["classification"] = UnitClassification,
    ["family"] = UnitCreatureFamily,
    ["name"] = UnitName,
    ["guild"] = GetGuildInfo,
    ["class"] = UnitClassBase,
    ["type"] = UnitCreatureType,
    ["faction"] = UnitFactionGroup,
    ["player or npc"] = function( str )
        return UnitIsPlayer( str ) and "player" or "npc";
    end,
    ["race"] = UnitRace,
    ["server"] = function( str )
        if( UnitIsPlayer( str ) ) then
            return select( 2, UnitName( str ) ) or active_server;
        end
    end };

-- palettize predicate table.  whether or not to palettize
palettize_p = {
    ["classification"] = true,
    ["family"] = true,
    ["server"] = true,
    --["name"] = true,
    --["guild"] = true,
    ["class"] = true,
    ["type"] = true,
    ["faction"] = true,
    ["player or npc"] = true,
    ["race"] = true,
    ["server"] = true
};

get_unit_info = function( str )
    local tbl = {};
    for name, fn in pairs( unit_properties ) do
        tbl[name] = fn( str );
    end

    return tbl;
end

-- [[ TARGETNOTES_DB structure ]]--
-- sequential, each element is an entry
-- each entry has 2 tables
-- table entry[1] has the unit profile (name, class, race, etc)
-- table entry[2] (sequential) has the notes associated with the unit
-- entry[1] is mostly palettized, but this is hidden from most of the addon

local get_palette_system = function()
    -- __index template
    -- automatically assigns incrementing numbers to new keys
    local inc_on_new = function( tbl, key )
        local count = 0;
        for key1 in pairs( tbl ) do
            count = count + 1;
        end

        count = count + 1;

        tbl[key] = count;
        return count;
    end

    -- __call template
    -- lazy inverse table, so we don't need to update anything when source table grows
    local lazy_inv = function( tbl1 )
        local tbl2 = {};
        return function( tbl, inv )
            for key, value in pairs( tbl1 ) do
                if( value == inv ) then
                    tbl2[inv] = key;
                    break;
                end
            end

            return tbl2[inv];
        end
    end

    -- grab existing palettes and remetafy them
    TARGETNOTES_DB.l1palette = TARGETNOTES_DB.l1palette or {};
    TARGETNOTES_DB.l2palette = TARGETNOTES_DB.l2palette or {};

    local l1p, l2p = TARGETNOTES_DB.l1palette, TARGETNOTES_DB.l2palette;

    setmetatable( l1p, { __index = inc_on_new, __call = lazy_inv( l1p ) } );

    for key, tbl in pairs( l2p ) do
        setmetatable( tbl, { __index = inc_on_new, __call = lazy_inv( tbl ) } );
    end

    -- metafy newcomers
    setmetatable( l2p, { __index = function( tbl, field )
        local temp = {};
        tbl[field] = setmetatable( temp, { __index = inc_on_new, __call = lazy_inv( temp ) } );

        return tbl[field];
    end } );

    local palettize = function( tbl )
        local profile = tbl[1];

        local collection = {};
        for field, value in pairs( profile ) do
            collection[l1p[field]] = palettize_p[field] and l2p[field][value] or value;
        end

        return { collection, tbl[2] };
    end

    local deuninpalettize = function( tbl )
        local profile = tbl[1];

        local collection = {};
        for field, value in pairs( profile ) do
            field = l1p(field);
            collection[field] = palettize_p[field] and l2p[field](value) or value;
        end

        return { collection, tbl[2] };
    end

    return palettize, deuninpalettize;
end

local last_target_profile;

local get_id = function( profile )
    local name, server = profile["name"], profile["server"];
    return DB:Find( name, function( tbl )
        return ( tbl[1]["name"] == name ) and ( tbl[1]["server"] == server );
    end );
end

local frm = CreateFrame( "frame" );

-- wait for variables to be loaded
frm:RegisterEvent( "ADDON_LOADED" );
frm:SetScript( "onevent", function( _, _, addon )
    if( addon ~= "Toki's TargetNotes" ) then return end; -- return

    active_server = GetRealmName();
    TARGETNOTES_FUNCTIONS["active_server"] = active_server;

    --[[ ## database setup ## ]]--

    TARGETNOTES_DB = TARGETNOTES_DB or {};
    TARGETNOTES_INDEX = TARGETNOTES_INDEX or {};

    local palettize, deuninpalettize = get_palette_system();

    DB = TARGETNOTES_FUNCTIONS["CreateDB"]{
        ["content database"] = TARGETNOTES_DB,
        ["index database"] = TARGETNOTES_INDEX,
        ["storage function"] = palettize,
        ["extraction function"] = deuninpalettize,
        ["minimum prefix length"] = 3,
        ["case sensitive?"] = false };

    frm:UnregisterEvent( "ADDON_LOADED" );

    -- the player is the default target
    last_target_profile = get_unit_info( "player" );
    -- correction for lag in determining whether player is a player when first loading into game world
    last_target_profile["server"] = active_server;

    --[[ ## target changed event setup ## ]]--

    frm:RegisterEvent( "PLAYER_TARGET_CHANGED" );
    frm:SetScript( "onevent", function()
        if( UnitExists( "target" ) ) then
            -- update profile when switching to a new target
            last_target_profile = get_unit_info( "target" );

            local id = get_id( last_target_profile );

            if( id ) then -- there's an entry for this target
                local notes = DB:GetEntry(id)[2];
                local len = #notes;
                local min = math.min( TARGETNOTES_SETTINGS["max notes to display"], len );

                highlight_notes_q();

                if( len == 0 ) then
                    NoteDisplay:SetText( "" );
                    Highlight:Hide();
                elseif( len == 1 ) then
                    NoteDisplay:SetText( " " .. notes[1] .. " " );
                else
                    NoteDisplay:SetText( " " .. table.concat( notes, " \n ", len - min + 1, len ) );
                end

                TARGETNOTES_FRAME:Show();
            else
                TARGETNOTES_FRAME:Hide();
            end
        else
            TARGETNOTES_FRAME:Hide();
        end
    end );
end );

local current_search_string = "";
local function search( search_string, raw )
    if not( raw ) then
        search_string = (search_string == "") and current_search_string or search_string;
    end

    current_search_string = search_string;

    local results = DB:Search( search_string );

    table.sort( results, function( a, b )
        return a["rank"] > b["rank"];
    end );

    local editbox = SearchWindow.search_edit_box;

    editbox:SetText( search_string );

    if( #results == 0 ) then
        editbox:SetFocus();
    end

    editbox:SetScript( "ontextchanged", function( self )
        search( self:GetText(), true );
    end );

    tabulate_data( results, DB, function() search( search_string ) end );

    return #results;
end

local function update_entry( profile, note )
    local id = get_id( profile );

    if( id ) then -- there's an entry for this target
        local notes = DB:GetEntry(id)[2];
        -- remove to unindex the old profile
        DB:RemoveEntry( id );
        
        -- reinsert with the new profile and note
        table.insert( notes, note );
        DB:AddEntry( id, { profile, notes } );
    else
        DB:AddEntry( { profile, { note } } ); -- append as a new entry
    end

    frm:GetScript( "onevent" )(); -- display the new note and refresh search
    search( current_search_string );

    -- ping the red thingie
                                      Ping:Show();
    timer:Schedule( .35, function( self ) Ping:Hide();
        self:Schedule( .05, function( self ) Ping:Show();
            self:Schedule( .30, function( self ) Ping:Hide();
            end );
        end );
    end );
end

--[[ ## slash command setup ## ]]--

SLASH_TARGETNOTES_tn1 = command1;
SLASH_TARGETNOTES_tn2 = command2;

-- notetaking
SlashCmdList["TARGETNOTES_tn"] = function( note )
    update_entry( last_target_profile, note );
end

-- searching
SLASH_TARGETNOTES_ts1 = search_command1;
SLASH_TARGETNOTES_ts2 = search_command2;

SlashCmdList["TARGETNOTES_ts"] = function( search_string )
    SearchWindow:Show();
    search( search_string );
end

TARGETNOTES_FUNCTIONS["update_target_notes"] = function() frm:GetScript( "onevent" )() end;

-- hotkey function to toggle search window
TARGETNOTES_FUNCTIONS["search:toggle"] = function()
    if( SearchWindow:IsVisible() ) then
        local close = SearchWindow.close_button;
        close:GetScript( "onclick" )( close );
    else
        local num = search( current_search_string );
        SearchWindow:Show()

        -- the hotkey is not devoured so it shows up in the edit box as soon
        -- as it is autofocused.  here we give some delay before it's autofocused.
        -- hopefully the user has released the key by then
        if( num == 0 ) then
            timer:Schedule( .1, function()
                SearchWindow.search_edit_box:SetFocus();
            end );
        end

    end
end
